package slimeknights.tconstruct.tools.inventory;

import com.google.common.collect.Lists;
import com.google.common.collect.Queues;
import com.google.common.collect.Sets;

import net.minecraft.block.state.IBlockState;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.GuiScreen;
import net.minecraft.inventory.IInventory;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.BlockPos;
import net.minecraft.world.World;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import org.apache.commons.lang3.tuple.Pair;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Queue;
import java.util.Set;

import slimeknights.mantle.inventory.ContainerMultiModule;
import slimeknights.tconstruct.tools.block.BlockToolTable;
import slimeknights.tconstruct.tools.block.ITinkerStationBlock;
import slimeknights.tconstruct.tools.client.GuiTinkerStation;

public class ContainerTinkerStation<T extends TileEntity & IInventory> extends ContainerMultiModule<T> {

  public final boolean hasCraftingStation;
  public final List<Pair<BlockPos, IBlockState>> tinkerStationBlocks;

  public ContainerTinkerStation(T tile) {
    super(tile);

    tinkerStationBlocks = Lists.newLinkedList();
    hasCraftingStation = detectedTinkerStationParts(tile.func_145831_w(), tile.func_174877_v());
  }

  @SuppressWarnings("unchecked")
  public <TE extends TileEntity> TE getTinkerTE(Class<TE> clazz) {
    for(Pair<BlockPos, IBlockState> pair : tinkerStationBlocks) {
      TileEntity te = this.world.func_175625_s(pair.getLeft());

      if(te != null && clazz.isAssignableFrom(te.getClass())) {
        return (TE) te;
      }
    }
    return null;
  }

  public boolean detectedTinkerStationParts(World world, BlockPos start) {
    Set<Integer> found = Sets.newHashSet();
    Set<BlockPos> visited = Sets.newHashSet();
    Set<IBlockState> ret = Sets.newHashSet();
    boolean hasMaster = false;

    // BFS for related blocks
    Queue<BlockPos> queue = Queues.newPriorityQueue();
    queue.add(start);

    while(!queue.isEmpty()) {
      BlockPos pos = queue.poll();
      // already visited between adding and call
      if(visited.contains(pos)) {
        continue;
      }

      IBlockState state = world.func_180495_p(pos);
      if(!(state.func_177230_c() instanceof ITinkerStationBlock)) {
        // not a valid block for us
        continue;
      }

      // found a part, add surrounding blocks that haven't been visited yet
      if(!visited.contains(pos.func_177978_c())) {
        queue.add(pos.func_177978_c());
      }
      if(!visited.contains(pos.func_177974_f())) {
        queue.add(pos.func_177974_f());
      }
      if(!visited.contains(pos.func_177968_d())) {
        queue.add(pos.func_177968_d());
      }
      if(!visited.contains(pos.func_177976_e())) {
        queue.add(pos.func_177976_e());
      }
      // add to visited
      visited.add(pos);

      // save the thing
      ITinkerStationBlock tinker = (ITinkerStationBlock) state.func_177230_c();
      Integer number = tinker.getGuiNumber(state);
      if(!found.contains(number)) {
        found.add(number);
        tinkerStationBlocks.add(Pair.of(pos, state));
        ret.add(state);
        if(state.func_177228_b().containsKey(BlockToolTable.TABLES)) {
          BlockToolTable.TableTypes type = (BlockToolTable.TableTypes) state.func_177229_b(BlockToolTable.TABLES);
          if(type != null && type == BlockToolTable.TableTypes.CraftingStation)
            hasMaster = true;
        }
      }
    }

    // sort the found blocks by priority
    TinkerBlockComp comp = new TinkerBlockComp();
    Collections.sort(tinkerStationBlocks, comp);

    /*
    if(!hasMaster || foundBlocks.size() < 2) {

      // sort all the blocks according to their number
      TinkerBlockComp comp = new TinkerBlockComp(world);
      foundBlocks.sort(comp);

      for(BlockPos pos : foundBlocks) {
        IBlockState state = world.getBlockState(pos);
        ItemStack stack = state.getBlock().getDrops(world, pos, state, 0).get(0);
        tinkerTabs.addTab(stack, pos);
      }
    }*/

    return hasMaster;
  }

  /** Tells the client to take the current state and update its info displays */
  public void updateGUI() {
    if(tile.func_145831_w().field_72995_K) {
      Minecraft.func_71410_x().func_152344_a(new Runnable() {
        @Override
        public void run() {
          ContainerTinkerStation.clientGuiUpdate();
        }
      });
    }
  }

  /** Tells the client to display the LOCALIZED error message */
  public void error(final String message) {
    if(tile.func_145831_w().field_72995_K) {
      Minecraft.func_71410_x().func_152344_a(new Runnable() {
        @Override
        public void run() {
          ContainerTinkerStation.clientError(message);
        }
      });
    }
  }

  /** Tells the client to display the LOCALIZED warning message */
  public void warning(final String message) {
    if(tile.func_145831_w().field_72995_K) {
      Minecraft.func_71410_x().func_152344_a(new Runnable() {
        @Override
        public void run() {
          ContainerTinkerStation.clientWarning(message);
        }
      });
    }
  }

  @SideOnly(Side.CLIENT)
  private static void clientGuiUpdate() {
    GuiScreen screen = Minecraft.func_71410_x().field_71462_r;
    if(screen instanceof GuiTinkerStation) {
      ((GuiTinkerStation) screen).updateDisplay();
    }
  }

  @SideOnly(Side.CLIENT)
  private static void clientError(String message) {
    GuiScreen screen = Minecraft.func_71410_x().field_71462_r;
    if(screen instanceof GuiTinkerStation) {
      ((GuiTinkerStation) screen).error(message);
    }
  }

  @SideOnly(Side.CLIENT)
  private static void clientWarning(String message) {
    GuiScreen screen = Minecraft.func_71410_x().field_71462_r;
    if(screen instanceof GuiTinkerStation) {
      ((GuiTinkerStation) screen).warning(message);
    }
  }

  private static class TinkerBlockComp implements Comparator<Pair<BlockPos, IBlockState>> {

    @Override
    public int compare(Pair<BlockPos, IBlockState> o1, Pair<BlockPos, IBlockState> o2) {
      IBlockState s1 = o1.getRight();
      IBlockState s2 = o2.getRight();

      return ((ITinkerStationBlock)s2.func_177230_c()).getGuiNumber(s2) - ((ITinkerStationBlock)s1.func_177230_c()).getGuiNumber(s1);
    }
  }
}
