package slimeknights.tconstruct.library.utils;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;

import net.minecraft.block.Block;
import net.minecraft.block.material.Material;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.Minecraft;
import net.minecraft.client.multiplayer.PlayerControllerMP;
import net.minecraft.enchantment.Enchantment;
import net.minecraft.enchantment.EnchantmentHelper;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.SharedMonsterAttributes;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.entity.player.InventoryPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.network.play.client.C07PacketPlayerDigging;
import net.minecraft.network.play.server.S0BPacketAnimation;
import net.minecraft.network.play.server.S12PacketEntityVelocity;
import net.minecraft.network.play.server.S23PacketBlockChange;
import net.minecraft.potion.Potion;
import net.minecraft.stats.AchievementList;
import net.minecraft.stats.StatList;
import net.minecraft.util.BlockPos;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.MathHelper;
import net.minecraft.util.MovingObjectPosition;
import net.minecraft.util.Vec3i;
import net.minecraft.world.World;
import net.minecraft.world.WorldServer;
import net.minecraftforge.common.ForgeHooks;
import net.minecraftforge.common.IShearable;

import java.util.List;

import slimeknights.tconstruct.TConstruct;
import slimeknights.tconstruct.common.TinkerNetwork;
import slimeknights.tconstruct.library.TinkerRegistry;
import slimeknights.tconstruct.library.tinkering.Category;
import slimeknights.tconstruct.library.tinkering.TinkersItem;
import slimeknights.tconstruct.library.tools.ToolCore;
import slimeknights.tconstruct.library.traits.ITrait;
import slimeknights.tconstruct.tools.events.TinkerToolEvent;
import slimeknights.tconstruct.tools.network.ToolBreakAnimationPacket;

public final class ToolHelper {

  private ToolHelper() {
  }

  public static boolean hasCategory(ItemStack stack, Category category) {
    if(stack == null || stack.func_77973_b() == null || !(stack.func_77973_b() instanceof TinkersItem)) {
      return false;
    }

    return ((TinkersItem) stack.func_77973_b()).hasCategory(category);
  }

  /* Basic Tool data */
  public static int getDurabilityStat(ItemStack stack) {
    return getIntTag(stack, Tags.DURABILITY);
  }

  public static int getHarvestLevelStat(ItemStack stack) {
    return getIntTag(stack, Tags.HARVESTLEVEL);
  }

  /** Returns the speed saved on the tool. NOT the actual mining speed, see getActualMiningSpeed */
  public static float getMiningSpeedStat(ItemStack stack) {
    return getfloatTag(stack, Tags.MININGSPEED);
  }

  public static float getAttackStat(ItemStack stack) {
    return getfloatTag(stack, Tags.ATTACK);
  }

  public static float getActualAttack(ItemStack stack) {
    float damage = getAttackStat(stack);
    if(stack != null && stack.func_77973_b() instanceof ToolCore) {
      damage *= ((ToolCore) stack.func_77973_b()).damagePotential();
    }
    return damage;
  }

  /** Returns the speed saved on the tool. NOT the actual mining speed, see getActualMiningSpeed */
  public static float getActualMiningSpeed(ItemStack stack) {
    float speed = getMiningSpeedStat(stack);
    if(stack != null && stack.func_77973_b() instanceof ToolCore) {
      speed *= ((ToolCore) stack.func_77973_b()).miningSpeedModifier();
    }
    return speed;
  }


  public static int getFreeModifiers(ItemStack stack) {
    return getIntTag(stack, Tags.FREE_MODIFIERS);
  }

  public static List<ITrait> getTraits(ItemStack stack) {
    List<ITrait> traits = Lists.newLinkedList();
    NBTTagList traitsTagList = TagUtil.getTraitsTagList(stack);
    for(int i = 0; i < traitsTagList.func_74745_c(); i++) {
      ITrait trait = TinkerRegistry.getTrait(traitsTagList.func_150307_f(i));
      if(trait != null) {
        traits.add(trait);
      }
    }
    return traits;
  }

  public static float calcDigSpeed(ItemStack stack, IBlockState blockState) {
    if(blockState == null) {
      return 0f;
    }

    if(!stack.func_77942_o()) {
      return 1f;
    }

    // check if the tool has the correct class and harvest level
    if(!canHarvest(stack, blockState)) {
      return 0f;
    }

    if(isBroken(stack)) {
      return 0.3f;
    }

    // calculate speed depending on stats

    // strength = default 1
    NBTTagCompound tag = TagUtil.getToolTag(stack);
    float strength = stack.func_77973_b().func_150893_a(stack, blockState.func_177230_c());
    float speed = tag.func_74760_g(Tags.MININGSPEED);

    if(stack.func_77973_b() instanceof ToolCore) {
      speed *= ((ToolCore) stack.func_77973_b()).miningSpeedModifier();
    }

    return strength * speed;
  }

  /**
   * Returns true if the tool is effective for harvesting the given block.
   */
  public static boolean isToolEffective(ItemStack stack, IBlockState state) {
    // check material
    for(String type : stack.func_77973_b().getToolClasses(stack)) {
      if(state.func_177230_c().isToolEffective(type, state)) {
        return true;
      }
    }

    return false;
  }

  // also checks for the tools effectiveness
  public static boolean isToolEffective2(ItemStack stack, IBlockState state) {
    if(isToolEffective(stack, state))
      return true;

    if(stack.func_77973_b() instanceof ToolCore && ((ToolCore) stack.func_77973_b()).isEffective(state.func_177230_c()))
      return true;

    return false;
  }

  /**
   * Checks if an item has the right harvest level of the correct type for the block.
   */
  public static boolean canHarvest(ItemStack stack, IBlockState state) {
    Block block = state.func_177230_c();

    // doesn't require a tool
    if(block.func_149688_o().func_76229_l()) {
      return true;
    }

    String type = block.getHarvestTool(state);
    int level = block.getHarvestLevel(state);

    return stack.func_77973_b().getHarvestLevel(stack, type) >= level;
  }

  /* Harvesting */



  public static ImmutableList<BlockPos> calcAOEBlocks(ItemStack stack, World world, EntityPlayer player, BlockPos origin, int width, int height, int depth) {
    return calcAOEBlocks(stack, world, player, origin, width, height, depth, -1);
  }

  public static ImmutableList<BlockPos> calcAOEBlocks(ItemStack stack, World world, EntityPlayer player, BlockPos origin, int width, int height, int depth, int distance) {
    // only works with toolcore because we need the raytrace call
    if(stack == null || !(stack.func_77973_b() instanceof ToolCore))
      return ImmutableList.of();

    // find out where the player is hitting the block
    IBlockState state = world.func_180495_p(origin);
    Block block = state.func_177230_c();

    if(block.func_149688_o() == Material.field_151579_a) {
      // what are you DOING?
      return ImmutableList.of();
    }

    MovingObjectPosition mop = ((ToolCore) stack.func_77973_b()).func_77621_a(world, player, false);
    if(mop == null) {
      return ImmutableList.of();
    }

    // fire event
    TinkerToolEvent.ExtraBlockBreak event = TinkerToolEvent.ExtraBlockBreak.fireEvent(stack, player, width, height, depth, distance);
    width = event.width;
    height = event.height;
    depth = event.depth;
    distance = event.distance;

    // we know the block and we know which side of the block we're hitting. time to calculate the depth along the different axes
    int x,y,z;
    BlockPos start = origin;
    switch(mop.field_178784_b) {
      case DOWN:
      case UP:
        // x y depends on the angle we look?
        Vec3i vec = player.func_174811_aO().func_176730_m();
        x = vec.func_177958_n() * height + vec.func_177952_p() * width;
        y = mop.field_178784_b.func_176743_c().func_179524_a() * -depth;
        z = vec.func_177958_n() * width + vec.func_177952_p() * height;
        start = start.func_177982_a(-x/2, 0, -z/2);
        if(x % 2 == 0) {
          if(x > 0 && mop.field_72307_f.field_72450_a - mop.func_178782_a().func_177958_n() > 0.5d) start = start.func_177982_a(1,0,0);
          else if (x < 0 && mop.field_72307_f.field_72450_a - mop.func_178782_a().func_177958_n() < 0.5d) start = start.func_177982_a(-1,0,0);
        }
        if(z % 2 == 0) {
          if(z > 0 && mop.field_72307_f.field_72449_c - mop.func_178782_a().func_177952_p() > 0.5d) start = start.func_177982_a(0,0,1);
          else if(z < 0 && mop.field_72307_f.field_72449_c - mop.func_178782_a().func_177952_p() < 0.5d) start = start.func_177982_a(0,0,-1);
        }
        break;
      case NORTH:
      case SOUTH:
        x = width;
        y = height;
        z = mop.field_178784_b.func_176743_c().func_179524_a() * -depth;
        start = start.func_177982_a(-x/2, -y/2, 0);
        if(x % 2 == 0 && mop.field_72307_f.field_72450_a - mop.func_178782_a().func_177958_n() > 0.5d) start = start.func_177982_a(1,0,0);
        if(y % 2 == 0 && mop.field_72307_f.field_72448_b - mop.func_178782_a().func_177956_o() > 0.5d) start = start.func_177982_a(0,1,0);
        break;
      case WEST:
      case EAST:
        x = mop.field_178784_b.func_176743_c().func_179524_a() * -depth;
        y = height;
        z = width;
        start = start.func_177982_a(-0, -y/2, -z/2);
        if(y % 2 == 0 && mop.field_72307_f.field_72448_b - mop.func_178782_a().func_177956_o() > 0.5d) start = start.func_177982_a(0,1,0);
        if(z % 2 == 0 && mop.field_72307_f.field_72449_c - mop.func_178782_a().func_177952_p() > 0.5d) start = start.func_177982_a(0,0,1);
        break;
      default:
        x = y = z = 0;
    }

    ImmutableList.Builder<BlockPos> builder = ImmutableList.builder();
    for(int xp = start.func_177958_n(); xp != start.func_177958_n() + x; xp += x/MathHelper.func_76130_a(x)) {
      for(int yp = start.func_177956_o(); yp != start.func_177956_o() + y; yp += y/MathHelper.func_76130_a(y)) {
        for(int zp = start.func_177952_p(); zp != start.func_177952_p() + z; zp += z/MathHelper.func_76130_a(z)) {
          // don't add the origin block
          if(xp == origin.func_177958_n() && yp == origin.func_177956_o() && zp == origin.func_177952_p()) {
            continue;
          }
          if(distance > 0 && MathHelper.func_76130_a(xp - origin.func_177958_n()) + MathHelper.func_76130_a(yp - origin.func_177956_o()) + MathHelper.func_76130_a(
              zp - origin.func_177952_p()) > distance) {
            continue;
          }
          BlockPos pos = new BlockPos(xp, yp, zp);
          if(isToolEffective2(stack, world.func_180495_p(pos))) {
            builder.add(pos);
          }
        }
      }
    }

    return builder.build();
  }

  public static void breakExtraBlock(ItemStack stack, World world, EntityPlayer player, BlockPos pos, BlockPos refPos) {
    // prevent calling that stuff for air blocks, could lead to unexpected behaviour since it fires events
    if (world.func_175623_d(pos))
      return;

    //if(!(player instanceof EntityPlayerMP)) {
      //return;
    //}

    // check if the block can be broken, since extra block breaks shouldn't instantly break stuff like obsidian
    // or precious ores you can't harvest while mining stone
    IBlockState state = world.func_180495_p(pos);
    Block block = state.func_177230_c();

    // only effective materials
    if(!isToolEffective2(stack, state)) {
      return;
    }

    IBlockState refState = world.func_180495_p(refPos);
    float refStrength = ForgeHooks.blockStrength(refState, player, world, refPos);
    float strength = ForgeHooks.blockStrength(state, player, world, pos);

    // only harvestable blocks that aren't impossibly slow to harvest
    if (!ForgeHooks.canHarvestBlock(block, player, world, pos) || refStrength/strength > 10f)
      return;

    // From this point on it's clear that the player CAN break the block

    if (player.field_71075_bZ.field_75098_d) {
      block.func_176208_a(world, pos, state, player);
      if (block.removedByPlayer(world, pos, player, false))
        block.func_176206_d(world, pos, state);

      // send update to client
      if (!world.field_72995_K) {
        ((EntityPlayerMP)player).field_71135_a.func_147359_a(new S23PacketBlockChange(world, pos));
      }
      return;
    }

    // callback to the tool the player uses. Called on both sides. This damages the tool n stuff.
    stack.func_179548_a(world, block, pos, player);

    // server sided handling
    if (!world.field_72995_K) {
      // send the blockbreak event
      int xp = ForgeHooks.onBlockBreakEvent(world, ((EntityPlayerMP) player).field_71134_c.func_73081_b(), (EntityPlayerMP) player, pos);
      if(xp == -1) {
        return;
      }


      // serverside we reproduce ItemInWorldManager.tryHarvestBlock

      // ItemInWorldManager.removeBlock
      block.func_176208_a(world, pos, state, player);

      if(block.removedByPlayer(world, pos, player, true)) // boolean is if block can be harvested, checked above
      {
        block.func_176206_d(world, pos, state);
        block.func_180657_a(world, player, pos, state, world.func_175625_s(pos));
        block.func_180637_b(world, pos, xp);
      }

      // always send block update to client
      EntityPlayerMP mpPlayer = (EntityPlayerMP) player;
      mpPlayer.field_71135_a.func_147359_a(new S23PacketBlockChange(world, pos));
    }
    // client sided handling
    else {
      PlayerControllerMP pcmp = Minecraft.func_71410_x().field_71442_b;
      // clientside we do a "this clock has been clicked on long enough to be broken" call. This should not send any new packets
      // the code above, executed on the server, sends a block-updates that give us the correct state of the block we destroy.

      // following code can be found in PlayerControllerMP.onPlayerDestroyBlock
      world.func_175718_b(2001, pos, Block.func_176210_f(state));
      if(block.removedByPlayer(world, pos, player, true))
      {
        block.func_176206_d(world, pos, state);
      }
      // callback to the tool
      stack.func_179548_a(world, block, pos, player);

      if (stack.field_77994_a == 0 && stack == player.func_71045_bC())
      {
        player.func_71028_bD();
      }

      // send an update to the server, so we get an update back
      //if(PHConstruct.extraBlockUpdates)
      Minecraft.func_71410_x().func_147114_u().func_147297_a(new C07PacketPlayerDigging(C07PacketPlayerDigging.Action.STOP_DESTROY_BLOCK, pos, Minecraft
          .func_71410_x().field_71476_x.field_178784_b));
    }
  }

  public static boolean shearBlock(ItemStack itemstack, World world, EntityPlayer player, BlockPos pos) {
    // only serverside since it creates entities
    if(world.field_72995_K) {
      return false;
    }

    Block block = world.func_180495_p(pos).func_177230_c();
    if (block instanceof IShearable)
    {
      IShearable target = (IShearable)block;
      if (target.isShearable(itemstack, world, pos))
      {
        int fortune = EnchantmentHelper.func_77506_a(Enchantment.field_77346_s.field_77352_x, itemstack);
        List<ItemStack> drops = target.onSheared(itemstack, world, pos, fortune);

        for(ItemStack stack : drops)
        {
          float f = 0.7F;
          double d  = (double)(TConstruct.random.nextFloat() * f) + (double)(1.0F - f) * 0.5D;
          double d1 = (double)(TConstruct.random.nextFloat() * f) + (double)(1.0F - f) * 0.5D;
          double d2 = (double)(TConstruct.random.nextFloat() * f) + (double)(1.0F - f) * 0.5D;
          EntityItem entityitem = new EntityItem(player.field_70170_p, (double)pos.func_177958_n() + d, (double)pos.func_177956_o() + d1, (double)pos.func_177952_p() + d2, stack);
          entityitem.func_174869_p();
          world.func_72838_d(entityitem);
        }

        itemstack.func_77972_a(1, player);
        player.func_71064_a(net.minecraft.stats.StatList.field_75934_C[Block.func_149682_b(block)], 1);

        world.func_175698_g(pos);

        return true;
      }
    }
    return false;
  }


  /* Tool Durability */

  public static int getCurrentDurability(ItemStack stack) {
    return stack.func_77958_k() - stack.func_77952_i();
  }

  /** Damages the tool. Entity is only needed in case the tool breaks for rendering the break effect. */
  public static void damageTool(ItemStack stack, int amount, EntityLivingBase entity) {
    if(amount == 0 || isBroken(stack))
      return;

    int actualAmount = amount;
    NBTTagList list = TagUtil.getTraitsTagList(stack);
    for(int i = 0; i < list.func_74745_c(); i++) {
      ITrait trait = TinkerRegistry.getTrait(list.func_150307_f(i));
      if(trait != null) {
        if(amount > 0) {
          actualAmount = trait.onToolDamage(stack, amount, actualAmount, entity);
        } else {
          actualAmount = trait.onToolHeal(stack, amount, actualAmount, entity);
        }
      }
    }

    // ensure we never deal more damage than durability
    actualAmount = Math.min(actualAmount, getCurrentDurability(stack));
    stack.func_77964_b(stack.func_77952_i() + actualAmount);

    if(getCurrentDurability(stack) == 0) {
      breakTool(stack, entity);
    }
  }

  public static void healTool(ItemStack stack, int amount, EntityLivingBase entity) {
    damageTool(stack, -amount, entity);
  }

  public static boolean isBroken(ItemStack stack) {
    return TagUtil.getToolTag(stack).func_74767_n(Tags.BROKEN);
  }

  public static void breakTool(ItemStack stack, EntityLivingBase entity) {
    NBTTagCompound tag = TagUtil.getToolTag(stack);
    tag.func_74757_a(Tags.BROKEN, true);
    TagUtil.setToolTag(stack, tag);

    if(entity instanceof EntityPlayerMP) {
      TinkerNetwork.sendTo(new ToolBreakAnimationPacket(stack), (EntityPlayerMP) entity);
    }
  }

  public static void repairTool(ItemStack stack, int amount) {
    // entity is optional, only needed for rendering break effect, never needed when repairing
    repairTool(stack, amount, null);
  }

  public static void repairTool(ItemStack stack, int amount, EntityLivingBase entity) {
    if(isBroken(stack)) {
      // ensure correct damage value
      stack.func_77964_b(stack.func_77958_k());

      // setItemDamage might break the tool again, so we do this afterwards
      NBTTagCompound tag = TagUtil.getToolTag(stack);
      tag.func_74757_a(Tags.BROKEN, false);
      TagUtil.setToolTag(stack, tag);
    }

    TinkerToolEvent.OnRepair.fireEvent(stack, amount);

    healTool(stack, amount, entity);
  }


  /* Dealing tons of damage */

  /**
   * Makes all the calls to attack an entity. Takes enchantments and potions and traits into account. Basically call this when a tool deals damage.
   * Most of this function is the same as {@link EntityPlayer#attackTargetEntityWithCurrentItem(Entity targetEntity)}
   */
  public static boolean attackEntity(ItemStack stack, ToolCore tool, EntityPlayer player, Entity targetEntity) {
    // todo: check how 1.9 does this and if we should steal it
    // nothing to do, no target?
    if(targetEntity == null || !targetEntity.func_70075_an() || targetEntity.func_85031_j(player) || !stack.func_77942_o()) {
      return false;
    }
    if(!(targetEntity instanceof EntityLivingBase)) {
      return false;
    }
    if(isBroken(stack)) {
      return false;
    }
    EntityLivingBase target = (EntityLivingBase) targetEntity;

    // traits on the tool
    List<ITrait> traits = Lists.newLinkedList();
    NBTTagList traitsTagList = TagUtil.getTraitsTagList(stack);
    for(int i = 0; i < traitsTagList.func_74745_c(); i++) {
      ITrait trait = TinkerRegistry.getTrait(traitsTagList.func_150307_f(i));
      if(trait != null) {
        traits.add(trait);
      }
    }

    // players base damage
    float baseDamage = (float)player.func_110148_a(SharedMonsterAttributes.field_111264_e).func_111126_e();

    // missing because not supported by tcon tools: vanilla damage enchantments, we have our own modifiers
    // missing because not supported by tcon tools: vanilla knockback enchantments, we have our own modifiers
    float baseKnockback = player.func_70051_ag() ? 1 : 0;

    // tool damage
    baseDamage += ToolHelper.getAttackStat(stack);
    baseDamage *= tool.damagePotential();

    // calculate if it's a critical hit
    boolean isCritical = player.field_70143_R > 0.0F && !player.field_70122_E && !player.func_70617_f_() && !player.func_70090_H() && !player.func_70644_a(Potion.field_76440_q) && player.field_70154_o == null && targetEntity instanceof EntityLivingBase;
    for(ITrait trait : traits) {
      if(trait.isCriticalHit(stack, player, target))
        isCritical = true;
    }

    // calculate actual damage
    float damage = baseDamage;
    for(ITrait trait : traits) {
      damage = trait.damage(stack, player, target, baseDamage, damage, isCritical);
    }

    // apply critical damage
    if(isCritical) {
      damage *= 1.5f;
    }

    // calculate cutoff
    damage = calcCutoffDamage(damage, tool.damageCutoff());

    // calculate actual knockback
    float knockback = baseKnockback;
    for(ITrait trait : traits) {
      knockback = trait.knockBack(stack, player, target, damage, baseKnockback, knockback, isCritical);
    }

    // missing because not supported by tcon tools: vanilla fire aspect enchantments, we have our own modifiers

    float oldHP = target.func_110143_aJ();

    double oldVelX = target.field_70159_w;
    double oldVelY = target.field_70181_x;
    double oldVelZ = target.field_70179_y;


    int hurtResistantTime = target.field_70172_ad;
    // deal the damage
    for(ITrait trait : traits) {
      trait.onHit(stack, player, target, damage, isCritical);
      // reset hurt reristant time
      target.field_70172_ad = hurtResistantTime;
    }
    boolean hit = tool.dealDamage(stack, player, target, damage);

    // did we hit?
    if(hit) {
      // actual damage dealt
      float damageDealt = oldHP - target.func_110143_aJ();

      // apply knockback modifier
      oldVelX = target.field_70159_w = oldVelX + (target.field_70159_w - oldVelX)*tool.knockback();
      oldVelY = target.field_70181_x = oldVelY + (target.field_70181_x - oldVelY)*tool.knockback()/3f;
      oldVelZ = target.field_70179_y = oldVelZ + (target.field_70179_y - oldVelZ)*tool.knockback();

      // apply knockback
      if(knockback > 0f) {
        double velX = -MathHelper.func_76126_a(player.field_70177_z * (float) Math.PI / 180.0F) * knockback * 0.5F;
        double velZ = MathHelper.func_76134_b(player.field_70177_z * (float)Math.PI / 180.0F) * knockback * 0.5F;
        targetEntity.func_70024_g(velX, 0.1d, velZ);

        // slow down player
        player.field_70159_w *= 0.6f;
        player.field_70179_y *= 0.6f;
        player.func_70031_b(false);
      }

      // Send movement changes caused by attacking directly to hit players.
      // I guess this is to allow better handling at the hit players side? No idea why it resets the motion though.
      if (targetEntity instanceof EntityPlayerMP && targetEntity.field_70133_I)
      {
        ((EntityPlayerMP)targetEntity).field_71135_a.func_147359_a(new S12PacketEntityVelocity(targetEntity));
        targetEntity.field_70133_I = false;
        targetEntity.field_70159_w = oldVelX;
        targetEntity.field_70181_x = oldVelY;
        targetEntity.field_70179_y = oldVelZ;
      }

      // vanilla critical callback
      if(isCritical) {
        player.func_71009_b(target);
      }

      // "magical" critical damage? (aka caused by modifiers)
      if(damage > baseDamage) {
        // this usually only displays some particles :)
        player.func_71047_c(targetEntity);
      }

      // vanilla achievement support :D
      if(damage >= 18f) {
        player.func_71029_a(AchievementList.field_75999_E);
      }

      player.func_130011_c(target);

      // we don't support vanilla thorns or antispider enchantments
      //EnchantmentHelper.applyThornEnchantments(target, player);
      //EnchantmentHelper.applyArthropodEnchantments(player, target);


      // call post-hit callbacks before reducing the durability
      for(ITrait trait : traits) {
        trait.afterHit(stack, player, target, damageDealt, isCritical, hit); // hit is always true
      }

      // damage the tool
      stack.func_77961_a(target, player);
      if(!player.field_71075_bZ.field_75098_d) {
        tool.reduceDurabilityOnHit(stack, player, damage);
      }

      player.func_71064_a(StatList.field_75951_w, Math.round(damage*10f));
      player.func_71020_j(0.3f);
    }

    return true;
  }

  public static float calcCutoffDamage(float damage, float cutoff) {
    float p = 1f;
    float d = damage;
    damage = 0f;
    while(d > cutoff) {
      damage += p * cutoff;
      // safety for ridiculous values
      if(p > 0.001f) {
        p *= 0.9f;
      }
      else {
        damage += p * cutoff * ((d/cutoff) - 1f);
        return damage;
      }
      d -= cutoff;
    }

    damage += p*d;

    return damage;
  }

  public static float getActualDamage(ItemStack stack, EntityPlayer player) {
    float damage = (float)player.func_110148_a(SharedMonsterAttributes.field_111264_e).func_111126_e();
    damage += ToolHelper.getActualAttack(stack);

    if(stack.func_77973_b() instanceof ToolCore) {
      damage = ToolHelper.calcCutoffDamage(damage, ((ToolCore) stack.func_77973_b()).damageCutoff());
    }

    return damage;
  }

  public static void swingItem(int speed, EntityLivingBase entity) {
    if (!entity.field_82175_bq || entity.field_110158_av >= 3 || entity.field_110158_av < 0)
    {
      entity.field_110158_av = Math.min(4, -1 + speed);
      entity.field_82175_bq = true;

      if (entity.field_70170_p instanceof WorldServer)
      {
        ((WorldServer)entity.field_70170_p).func_73039_n().func_151247_a(entity, new S0BPacketAnimation(entity, 0));
      }
    }
  }

  public static boolean useSecondaryItem(ItemStack stack, EntityPlayer player, World world, BlockPos pos, EnumFacing side, float hitX, float hitY, float hitZ) {
    int slot = getSecondaryItemSlot(player);

    // last slot selected
    if(slot == player.field_71071_by.field_70461_c) {
      return false;
    }

    ItemStack secondaryItem = player.field_71071_by.func_70301_a(slot);

    // do we have an item to use?
    if(secondaryItem == null) {
      return false;
    }

    // use it
    int oldSlot = player.field_71071_by.field_70461_c;
    player.field_71071_by.field_70461_c = slot;
    boolean ret = secondaryItem.func_179546_a(player, world, pos, side, hitX, hitY, hitZ);
    // might have gotten used up
    if(secondaryItem.field_77994_a == 0) {
      player.field_71071_by.func_70299_a(slot, null);
    }
    player.field_71071_by.field_70461_c = oldSlot;

    return ret;
  }

  public static int getSecondaryItemSlot(EntityPlayer player) {
    int slot = player.field_71071_by.field_70461_c;
    int max = InventoryPlayer.func_70451_h() - 1;
    if(slot < max) {
      slot++;
    }

    // find next slot that has an item in it
    for(; slot < max; slot++) {
      ItemStack secondaryItem = player.field_71071_by.func_70301_a(slot);
      if(secondaryItem != null) {
        if(!(secondaryItem.func_77973_b() instanceof ToolCore) || !((ToolCore) secondaryItem.func_77973_b()).canUseSecondaryItem()) {
          break;
        }
      }
    }

    ItemStack secondaryItem = player.field_71071_by.func_70301_a(slot);
    if(secondaryItem != null) {
      if((secondaryItem.func_77973_b() instanceof ToolCore) && ((ToolCore) secondaryItem.func_77973_b()).canUseSecondaryItem()) {
        return player.field_71071_by.field_70461_c;
      }
    }

    return slot;
  }


  /* Helper Functions */

  static int getIntTag(ItemStack stack, String key) {
    NBTTagCompound tag = TagUtil.getToolTag(stack);

    return tag.func_74762_e(key);
  }

  static float getfloatTag(ItemStack stack, String key) {
    NBTTagCompound tag = TagUtil.getToolTag(stack);

    return tag.func_74760_g(key);
  }
}
