package mezz.jei.common.render;

import com.google.common.base.Preconditions;
import com.mojang.blaze3d.systems.RenderSystem;
import mezz.jei.api.ingredients.IIngredientRenderer;
import mezz.jei.api.ingredients.IIngredientTypeWithSubtypes;
import mezz.jei.common.Internal;
import mezz.jei.common.ingredients.RegisteredIngredients;
import mezz.jei.common.platform.IPlatformFluidHelperInternal;
import mezz.jei.common.util.ErrorUtil;
import net.minecraft.class_1058;
import net.minecraft.class_1159;
import net.minecraft.class_124;
import net.minecraft.class_1723;
import net.minecraft.class_1836;
import net.minecraft.class_2561;
import net.minecraft.class_287;
import net.minecraft.class_289;
import net.minecraft.class_290;
import net.minecraft.class_293;
import net.minecraft.class_3611;
import net.minecraft.class_3612;
import net.minecraft.class_4587;
import net.minecraft.class_5250;
import net.minecraft.class_757;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;

public class FluidTankRenderer<T> implements IIngredientRenderer<T> {
	private static final Logger LOGGER = LogManager.getLogger();

	private static final NumberFormat nf = NumberFormat.getIntegerInstance();
	private static final int TEXTURE_SIZE = 16;
	private static final int MIN_FLUID_HEIGHT = 1; // ensure tiny amounts of fluid are still visible

	private final IPlatformFluidHelperInternal<T> fluidHelper;
	private final long capacity;
	private final TooltipMode tooltipMode;
	private final int width;
	private final int height;

	enum TooltipMode {
		SHOW_AMOUNT,
		SHOW_AMOUNT_AND_CAPACITY,
		ITEM_LIST
	}

	public FluidTankRenderer(IPlatformFluidHelperInternal<T> fluidHelper) {
		this(fluidHelper, fluidHelper.bucketVolume(), TooltipMode.ITEM_LIST, 16, 16);
	}

	public FluidTankRenderer(IPlatformFluidHelperInternal<T> fluidHelper, long capacity, boolean showCapacity, int width, int height) {
		this(fluidHelper, capacity, showCapacity ? TooltipMode.SHOW_AMOUNT_AND_CAPACITY : TooltipMode.SHOW_AMOUNT, width, height);
	}

	private FluidTankRenderer(IPlatformFluidHelperInternal<T> fluidHelper, long capacity, TooltipMode tooltipMode, int width, int height) {
		Preconditions.checkArgument(capacity > 0, "capacity must be > 0");
		Preconditions.checkArgument(width > 0, "width must be > 0");
		Preconditions.checkArgument(height > 0, "height must be > 0");
		this.fluidHelper = fluidHelper;
		this.capacity = capacity;
		this.tooltipMode = tooltipMode;
		this.width = width;
		this.height = height;
	}

	@Override
	public void render(class_4587 poseStack, T fluidStack) {
		RenderSystem.enableBlend();

		drawFluid(poseStack, width, height, fluidStack);

		RenderSystem.setShaderColor(1, 1, 1, 1);

		RenderSystem.disableBlend();
	}

	private void drawFluid(class_4587 poseStack, final int width, final int height, T fluidStack) {
		IIngredientTypeWithSubtypes<class_3611, T> type = fluidHelper.getFluidIngredientType();
		class_3611 fluid = type.getBase(fluidStack);
		if (fluid.method_15780(class_3612.field_15906)) {
			return;
		}

		class_1058 fluidStillSprite = fluidHelper.getStillFluidSprite(fluidStack);
		if (fluidStillSprite == null) {
			return;
		}

		int fluidColor = fluidHelper.getColorTint(fluidStack);

		long amount = fluidHelper.getAmount(fluidStack);
		long scaledAmount = (amount * height) / capacity;
		if (amount > 0 && scaledAmount < MIN_FLUID_HEIGHT) {
			scaledAmount = MIN_FLUID_HEIGHT;
		}
		if (scaledAmount > height) {
			scaledAmount = height;
		}

		drawTiledSprite(poseStack, width, height, fluidColor, scaledAmount, fluidStillSprite);
	}

	private static void drawTiledSprite(class_4587 poseStack, final int tiledWidth, final int tiledHeight, int color, long scaledAmount, class_1058 sprite) {
		RenderSystem.setShaderTexture(0, class_1723.field_21668);
		class_1159 matrix = poseStack.method_23760().method_23761();
		setGLColorFromInt(color);

		final int xTileCount = tiledWidth / TEXTURE_SIZE;
		final int xRemainder = tiledWidth - (xTileCount * TEXTURE_SIZE);
		final long yTileCount = scaledAmount / TEXTURE_SIZE;
		final long yRemainder = scaledAmount - (yTileCount * TEXTURE_SIZE);

		final int yStart = tiledHeight;

		for (int xTile = 0; xTile <= xTileCount; xTile++) {
			for (int yTile = 0; yTile <= yTileCount; yTile++) {
				int width = (xTile == xTileCount) ? xRemainder : TEXTURE_SIZE;
				long height = (yTile == yTileCount) ? yRemainder : TEXTURE_SIZE;
				int x = (xTile * TEXTURE_SIZE);
				int y = yStart - ((yTile + 1) * TEXTURE_SIZE);
				if (width > 0 && height > 0) {
					long maskTop = TEXTURE_SIZE - height;
					int maskRight = TEXTURE_SIZE - width;

					drawTextureWithMasking(matrix, x, y, sprite, maskTop, maskRight, 100);
				}
			}
		}
	}

	private static void setGLColorFromInt(int color) {
		float red = (color >> 16 & 0xFF) / 255.0F;
		float green = (color >> 8 & 0xFF) / 255.0F;
		float blue = (color & 0xFF) / 255.0F;
		float alpha = ((color >> 24) & 0xFF) / 255F;

		RenderSystem.setShaderColor(red, green, blue, alpha);
	}

	private static void drawTextureWithMasking(class_1159 matrix, float xCoord, float yCoord, class_1058 textureSprite, long maskTop, long maskRight, float zLevel) {
		float uMin = textureSprite.method_4594();
		float uMax = textureSprite.method_4577();
		float vMin = textureSprite.method_4593();
		float vMax = textureSprite.method_4575();
		uMax = uMax - (maskRight / 16F * (uMax - uMin));
		vMax = vMax - (maskTop / 16F * (vMax - vMin));

		RenderSystem.setShader(class_757::method_34542);

		class_289 tessellator = class_289.method_1348();
		class_287 bufferBuilder = tessellator.method_1349();
		bufferBuilder.method_1328(class_293.class_5596.field_27382, class_290.field_1585);
		bufferBuilder.method_22918(matrix, xCoord, yCoord + 16, zLevel).method_22913(uMin, vMax).method_1344();
		bufferBuilder.method_22918(matrix, xCoord + 16 - maskRight, yCoord + 16, zLevel).method_22913(uMax, vMax).method_1344();
		bufferBuilder.method_22918(matrix, xCoord + 16 - maskRight, yCoord + maskTop, zLevel).method_22913(uMax, vMin).method_1344();
		bufferBuilder.method_22918(matrix, xCoord, yCoord + maskTop, zLevel).method_22913(uMin, vMin).method_1344();
		tessellator.method_1350();
	}

	@Override
	public List<class_2561> getTooltip(T fluidStack, class_1836 tooltipFlag) {
		IIngredientTypeWithSubtypes<class_3611, T> type = fluidHelper.getFluidIngredientType();
		class_3611 fluidType = type.getBase(fluidStack);
		try {
			if (fluidType.method_15780(class_3612.field_15906)) {
				return new ArrayList<>();
			}

			List<class_2561> tooltip = fluidHelper.getTooltip(fluidStack, tooltipFlag);

			long amount = fluidHelper.getAmount(fluidStack);
			long milliBuckets = (amount * 1000) / fluidHelper.bucketVolume();

			if (tooltipMode == TooltipMode.SHOW_AMOUNT_AND_CAPACITY) {
				class_5250 amountString = class_2561.method_43469("jei.tooltip.liquid.amount.with.capacity", nf.format(milliBuckets), nf.format(capacity));
				tooltip.add(amountString.method_27692(class_124.field_1080));
			} else if (tooltipMode == TooltipMode.SHOW_AMOUNT) {
				class_5250 amountString = class_2561.method_43469("jei.tooltip.liquid.amount", nf.format(milliBuckets));
				tooltip.add(amountString.method_27692(class_124.field_1080));
			}
			return tooltip;
		} catch (RuntimeException e) {
			RegisteredIngredients registeredIngredients = Internal.getRegisteredIngredients();
			String info = ErrorUtil.getIngredientInfo(fluidStack, type, registeredIngredients);
			LOGGER.error("Failed to get tooltip for fluid: " + info, e);
		}

		return new ArrayList<>();
	}

	@Override
	public int getWidth() {
		return width;
	}

	@Override
	public int getHeight() {
		return height;
	}
}
