package mezz.jei.api.recipe.transfer;

import org.jetbrains.annotations.Nullable;

import mezz.jei.api.gui.ingredient.IRecipeSlotsView;
import mezz.jei.api.gui.IRecipeLayout;
import mezz.jei.api.registration.IRecipeTransferRegistration;
import net.minecraft.class_1657;
import net.minecraft.class_1703;
import net.minecraft.class_2960;

/**
 * A recipe transfer handler moves items into a crafting area, based on the items in a recipe.
 *
 * Implementing this interface gives full control over the recipe transfer process.
 * Mods that use a regular slotted inventory can use {@link IRecipeTransferInfo} instead, which is much simpler.
 *
 * Useful functions for implementing a recipe transfer handler can be found in {@link IRecipeTransferHandlerHelper}.
 *
 * To register your recipe transfer handler, use {@link IRecipeTransferRegistration#addRecipeTransferHandler(IRecipeTransferHandler, class_2960)}
 */
public interface IRecipeTransferHandler<C extends class_1703, R> {
	/**
	 * The container that this recipe transfer handler can use.
	 */
	Class<C> getContainerClass();

	/**
	 * The recipe that this recipe transfer handler can use.
	 */
	Class<R> getRecipeClass();

	/**
	 * @param container   the container to act on
	 * @param recipe      the raw recipe instance
	 * @param recipeSlots the view of the recipe slots, with information about the ingredients
	 * @param player      the player, to do the slot manipulation
	 * @param maxTransfer if true, transfer as many items as possible. if false, transfer one set
	 * @param doTransfer  if true, do the transfer. if false, check for errors but do not actually transfer the items
	 * @return a recipe transfer error if the recipe can't be transferred. Return null on success.
	 *
	 * @since 9.3.0
	 */
	@Nullable
	default IRecipeTransferError transferRecipe(C container, R recipe, IRecipeSlotsView recipeSlots, class_1657 player, boolean maxTransfer, boolean doTransfer) {
		throw new UnsupportedOperationException("The new transferRecipe method has not been implemented on this recipe transfer handler");
	}

	/**
	 * @param container    the container to act on
	 * @param recipe       the raw recipe instance
	 * @param recipeLayout the layout of the recipe, with information about the ingredients
	 * @param player       the player, to do the slot manipulation
	 * @param maxTransfer  if true, transfer as many items as possible. if false, transfer one set
	 * @param doTransfer   if true, do the transfer. if false, check for errors but do not actually transfer the items
	 * @return a recipe transfer error if the recipe can't be transferred. Return null on success.
	 *
	 * @since 7.1.3
	 *
	 * @deprecated Use {@link #transferRecipe(class_1703, Object, IRecipeSlotsView, class_1657, boolean, boolean)} instead.
	 */
	@SuppressWarnings("removal")
	@Deprecated(forRemoval = true, since = "9.3.0")
	@Nullable
	default IRecipeTransferError transferRecipe(C container, R recipe, IRecipeLayout recipeLayout, class_1657 player, boolean maxTransfer, boolean doTransfer) {
		return null;
	}
}
