package slimeknights.mantle.data.loadable.array;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import it.unimi.dsi.fastutil.shorts.Short2ObjectFunction;
import net.minecraft.network.FriendlyByteBuf;
import slimeknights.mantle.data.loadable.Loadable;
import slimeknights.mantle.util.typed.TypedMap;

/** Loadable for a short array */
public record ShortArrayLoadable<T extends Number>(Loadable<T> base, int minSize, int maxSize, Short2ObjectFunction<T> mapper) implements ArrayLoadable.SizeRange<short[]> {
  @Override
  public int getLength(short[] array) {
    return array.length;
  }

  @Override
  public short[] convertCompact(JsonElement element, String key, TypedMap context) {
    return new short[] { base.convert(element, key, context).shortValue() };
  }

  @Override
  public short[] convertArray(JsonArray array, String key, TypedMap context) {
    short[] result = new short[array.size()];
    for (int i = 0; i < result.length; i++) {
      result[i] = base.convert(array.get(i), key + '[' + i + ']', context).shortValue();
    }
    return result;
  }

  @Override
  public JsonElement serializeFirst(short[] object) {
    return base.serialize(mapper.get(object[0]));
  }

  @Override
  public void serializeAll(JsonArray array, short[] object) {
    for (short element : object) {
      array.add(base.serialize(mapper.get(element)));
    }
  }

  @Override
  public short[] decode(FriendlyByteBuf buffer, TypedMap context) {
    int max = buffer.readVarInt();
    short[] array = new short[max];
    for (int i = 0; i < max; i++) {
      array[i] = base.decode(buffer, context).shortValue();
    }
    return array;
  }

  @Override
  public void encode(FriendlyByteBuf buffer, short[] array) {
    buffer.writeVarInt(array.length);
    for (short element : array) {
      base.encode(buffer, mapper.get(element));
    }
  }
}
