package slimeknights.mantle.command;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.mojang.brigadier.exceptions.DynamicCommandExceptionType;
import net.minecraft.SharedConstants;
import net.minecraft.network.chat.ClickEvent;
import net.minecraft.network.chat.ClickEvent.Action;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.packs.PackType;
import net.minecraft.world.level.storage.LevelResource;
import net.minecraftforge.common.crafting.CraftingHelper;
import net.minecraftforge.common.crafting.conditions.FalseCondition;
import net.minecraftforge.common.crafting.conditions.ICondition;
import net.minecraftforge.fml.ModList;
import slimeknights.mantle.Mantle;
import slimeknights.mantle.util.JsonHelper;

import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

/** Helpers for commands generating packs */
public class GeneratePackHelper {
  private static final String PACK_NAME = "SlimeKnightsGenerated";
  private static final String PACK_DESCRIPTION = "Data pack generated by commands from SlimeKnights mods.";
  /** Error when failing to save a single JSON */
  public static final DynamicCommandExceptionType FAILED_SAVE = new DynamicCommandExceptionType(id -> Mantle.makeComponent("command", "remove.failed_save", id));

  private GeneratePackHelper() {}

  /** Gets the path to the datapack */
  public static Path getDatapackPath(MinecraftServer server, String packName) {
    // if we have JSON Things, do a global datapack
    if (ModList.get().isLoaded("jsonthings")) {
      return server.getServerDirectory().toPath().resolve("thingpacks/" + packName);
    }
    // TODO: consider option to put in the standard datapacks folder via config property
    // otherwise, do a world local datapack
    return server.getWorldPath(LevelResource.DATAPACK_DIR).resolve(packName);
  }

  /** Gets the path to the datapack */
  public static Path getDatapackPath(MinecraftServer server) {
    return getDatapackPath(server, PACK_NAME);
  }

  /**
   * Saves the given JSON element
   * @param path     JSON to save
   * @param element  Path to save it
   */
  @SuppressWarnings("UnusedReturnValue")
  public static boolean saveJson(JsonElement element, Path path) {
    try {
      Files.createDirectories(path.getParent());
      String json = JsonHelper.DEFAULT_GSON.toJson(element);
      try (BufferedWriter bufferedwriter = Files.newBufferedWriter(path)) {
        bufferedwriter.write(json);
      }
      return true;
    } catch (IOException e) {
      Mantle.logger.error("Couldn't create JSON at {}", path, e);
      return false;
    }
  }

  /** Saves a JSON that removes the given resource using forge conditions */
  public static boolean saveConditionRemove(Path path, String conditionKey) {
    JsonObject json = new JsonObject();
    json.add(conditionKey, CraftingHelper.serialize(new ICondition[]{FalseCondition.INSTANCE}));
    return saveJson(json, path);
  }

  /** Saves a JSON that removes the given resource using forge conditions */
  public static boolean saveConditionRemove(Path path) {
    return saveConditionRemove(path, "forge:conditions");
  }

  /** Creates a mcmeta to make a valid pack */
  public static void saveMcmeta(Path folder, PackType packType, String description) {
    Path path = folder.resolve("pack.mcmeta");
    JsonObject meta = new JsonObject();
    JsonObject pack = new JsonObject();
    pack.addProperty("description", description);
    pack.addProperty("pack_format", SharedConstants.getCurrentVersion().getPackVersion(packType));
    meta.add("pack", pack);
    saveJson(meta, path);
  }

  /** Creates a mcmeta to make the server datapack */
  public static void saveMcmeta(Path folder) {
    saveMcmeta(folder, PackType.SERVER_DATA, PACK_DESCRIPTION);
  }

  /**
   * Makes a clickable text component for the output folder
   * @param file  File
   * @return  Clickable text component
   */
  public static Component getOutputComponent(File file) {
    return getOutputComponent(file.getAbsolutePath());
  }

  /**
   * Makes a clickable text component for the output folder
   * @param path  Path to log
   * @return  Clickable text component
   */
  public static Component getOutputComponent(Path path) {
    return getOutputComponent(path.toString());
  }

  /**
   * Makes a clickable text component for the output folder
   * @param path  Path to log
   * @return  Clickable text component
   */
  public static Component getOutputComponent(String path) {
    return getPathComponent(Component.literal(path), path);
  }

  /**
   * Makes a clickable text component for the output folder
   * @param text  Text to display
   * @param path  Path to log
   * @return  Clickable text component
   */
  public static MutableComponent getPathComponent(MutableComponent text, String path) {
    return text.withStyle(style -> style.withUnderlined(true).withClickEvent(new ClickEvent(Action.OPEN_FILE, path)));
  }
}
